<?php
/**
 * JIncludes is a Joomla! plugin allowing administrators to define code snippets 
 * and include/execute them inside a content item.
 *
 * PHP version 5
 *
 * @category   Plugin
 * @package    Joomla
 * @subpackage Content
 * @author     Andrea Azzini <andreazzini@gmail.com>
 * @copyright  2008 - 2012 Andrea Azzini
 * @license    GNU GPL v.3 or later, see http://www.gnu.org/licenses/
 * @version    1.0 (SVN: $Id: JIncludes.php 69 2012-01-25 23:17:23Z endi $)
 * @link       http://joomlacode.org/gf/project/jincludes
 *
 * This is a derivative work of the Joomla! Open Source CMS, 
 * see http://www.joomla.org/
 * Joomla! is licensed under the terms of the GNU GPL v.2 license.
 * 
 * This code is compatible with Joomla! versions 1.5, 1.6, 1.7, 2.5.
 * Separate packaging for 1.5 and 1.6+ is due to different naming conventions only.
 *
 * The plugin's usage is documented in the project's Wiki.
 * See http://joomlacode.org/gf/project/jincludes/wiki/
 */

// no direct access
defined('_JEXEC') or die();

jimport('joomla.plugin.plugin');


/**
 * The plugin is implemented as a derived class of JPlugin.
 *
 * @category Plugin
 * @package  JIncludes
 * @author   Andrea Azzini <andreazzini@gmail.com>
 * @license  GNU GPL v.3 or later, see http://www.gnu.org/licenses/
 * @link     http://joomlacode.org/gf/project/jincludes
 */
class plgContentJIncludes extends JPlugin
{
	
	/**
	 * Constructor. Loads the plugin's configuration from the db.
	 *
	 * @param object &$subject The object to observe
	 * @param object $params   The object that holds the plugin parameters
	 *
	 * @return void
	 */
	function __construct ( &$subject, $config )
	{
		parent::__construct($subject, $config);
		if (!($this->params)) { // Joomla 1.5 only
			$plugin =& JPluginHelper::getPlugin('content', 'JIncludes');
			$this->params = new JParameter( $plugin->params );
		}
		$this->block_types = array('phpblock_ext', 'phpblock');
		($this->num_snippets = (int)$this->params->get('num_snippets', '30')) or
			($this->num_snippets = 30);
		$this->show_errors = ( $this->params->get('show_errors', 'no') == 'yes' );
		
		if (version_compare('1.5.3', JVERSION, '>'))
			JError::raiseError(0, 'Joomla! versions older than 1.5.4 are not supported by JIncludes. Please update Joomla! or disable the plugin.');
	}
	
	/**
	 * Main event handler for JIncludes under Joomla 1.5. It is included as
	 * a compatibility for the new main event handler below, which uses the
	 * Joomla! 1.6 syntax.
	 *
	 * @param string &$article   the article the plugin is working on
	 * @param string &$params    currently unsupported
	 * @param int    $limitstart currently unsupported
	 *
	 * @return void
	 */
	public function onPrepareContent( &$article, &$params, $limitstart=0 )
	{
		$result = $this->onContentPrepare( 'JIncludesCompat', $article, $params, $limitstart );
		return $result;			
	}
	
	/**
	 * Main event handler for JIncludes. It is called directly by Joomla! 1.6
	 * or by the 1.5 compatibility function above.
	 *
	 * @param string $context    context string sent by the application
	 * @param string &$article   the article the plugin is working on
	 * @param string &$params    currently unsupported
	 * @param int    $page       currently unsupported
	 *
	 * @return boolean
	 */
	public function onContentPrepare( $context, &$article, &$params, $page=0 )
	{
		$this->keys       = array();
		$this->types      = array();
		$preg_block_keys  = '';
		$preg_single_keys = '';
		
		if (is_string($article))
			$target =& $article;
		elseif (property_exists($article,'text')) {
			$target =& $article->text;
			$this->_article = $article;
		} elseif (property_exists($article,'introtext'))
			$target =& $article->introtext;
		else
			return;
		
		for ($i=1; $i<=($this->num_snippets); $i++) {
			if ( ($key = $this->params->get("key$i", ''))
				&& (preg_match('#^/?[a-z\d_]+$#i', $key)) // check key name validity
				&& (!in_array($key, $this->keys))
			) {
				$this->keys[$i]  = $key;
				$this->types[$i] = $this->params->get("type$i", '');
				if (in_array($this->types[$i], $this->block_types)) {
					$preg_block_keys .= preg_quote($this->keys[$i], '#') . '|';
				} else {
					$preg_single_keys .= preg_quote($this->keys[$i], '#') . '|';
				}
			}
		}
		
		if ($preg_single_keys) {
			$preg_single_keys
				= '#{{\s*(' .substr($preg_single_keys, 0, -1).
				')\s*((?:\s.*?)?)}}#is';
			$target = preg_replace_callback(
				$preg_single_keys, array($this, '_callbackMatch'), $target
			);
		}
		if ($preg_block_keys) { 
			// this regex will always match the whole document as $matches[0]
			$preg_block_keys = '#^(.*?){{\s*(' .substr($preg_block_keys, 0, -1).
				')\s*((?:\s.*?)?)}}(.*?){{/\2}}(.*)$#is';
			do {
				$target = preg_replace_callback(
					$preg_block_keys, array($this, '_callbackMatch'),
					$target, -1, $found
				);
			} while ($found);
		}
		return true;
	}
	
	/**
	 * Recognizes a regexp match from the main event handler function.
	 * Only called by preg_replace_callback.
	 *
	 * @param array $matches the snippet's tag as tokenized by the matching regex
	 *
	 * @return string the snippet's output, to replace the tag in the article
	 */
	protected function _callbackMatch($matches)
	{
		if ( isset($matches[3]) ) { // = if block matching was performed
			$id    = array_search($matches[2], $this->keys);
			$env   = array( $matches[0], $matches[1], $matches[4], $matches[5] );
			$param = trim($matches[3]);
		} else {
			$id    = array_search($matches[1], $this->keys);
			$env   = array( $matches[0] );
			$param = trim($matches[2]);
		}
		$output = '';
		switch ($this->types[$id]) {
			case 'html':
				$output = $this->_renderHTML($id, $param);
				break;
			case 'html_ext':
				$output = $this->_renderExtHTML($id, $param, $matches[0]);
				break;
			case 'php':
			case 'phpblock':
				$output = $this->_renderPHP($id, $param, $env);
				break;
			case 'php_ext':
			case 'phpblock_ext':
				$output = $this->_renderExtPHP($id, $param, $env);
				break;
			case 'style_ext':
				$this->_attachStylesheet($id);
				break;
			case 'script_ext':
				$this->_attachScript($id);
				break;
			default: // found an unknown tag: don't touch the article
				$output = $matches[0];
		}
		return $output;
	}
	
	/*
	  HERE BEGIN the specific replace functions.
	  Remember: consistency checks are implemented before. When these functions 
	  are called, it is safe to assume that there is a snippet of the right type 
	  with ID=$id.
	  These functions return the markup to output in place of the snippet tag.
	 */
	
	/**
	 * Renders a snippet of type HTML (outputs the snippet's code)
	 *
	 * @param int    $id    the snippet being rendered
	 * @param string $param the inline parameter string
	 *
	 * @return string
	 */
	protected function _renderHTML( $id, $param )
	{
		$output = $this->params->get("code$id", '');
		if ($param) {
			$output = str_replace(
				'\$', '$', preg_replace('/([^\\\])\$/e', "'\\1'.\$param", $output)
			);
		}
		return $output;
	}
	
	/**
	 * Renders a snippet of type External HTML (outputs the contents of a file)
	 *
	 * @param int    $id      the snippet being rendered
	 * @param string $param   the inline parameter string
	 * @param string $default the snippet's tag, to be returned as a fallback
	 *
	 * @return string
	 */
	protected function _renderExtHTML( $id, $param, $default )
	{
		$filename = trim($this->params->get("code$id", ''));
		try {
			$output = file_get_contents($filename);
		}
		catch (Exception $e) { // we should not let an exception kill Joomla!
			ob_end_clean();
			if ($this->show_errors) {
				return '<strong>JIncludes (_renderExtHTML) caught exception: ' 
					.$e->getMessage(). '<strong>';
			} else {
				return $default;
			}
		}
		if ($param) {
			$output = str_replace(
				'\$', '$', preg_replace('/([^\\\])\$/e', "'\\1'.\$param", $output)
			);
		}
		return $output;
	}
	
	/**
	 * Renders a snippet of type PHP (eval's the entered code)
	 *
	 * @param int    $id    the snippet being rendered
	 * @param string $param the inline parameter string
	 * @param array  $env   the divided article (block mat.) or the snippet's tag
	 *
	 * @return string
	 */
	protected function _renderPHP( $id, $param, $env )
	{
		$code = $this->params->get("code$id", '');
		ob_start();
		eval( $code );
		$output = ob_get_contents();
		ob_end_clean();
		return $output;
	}
	
	/**
	 * Renders a snippet of type External PHP (includes a PHP file)
	 *
	 * @param int    $id    the snippet being rendered
	 * @param string $param the inline parameter string
	 * @param array  $env   the divided article (block mat.) or the snippet's tag
	 *
	 * @return string
	 */
	protected function _renderExtPHP( $id, $param, $env )
	{
		$filename = trim($this->params->get("code$id", ''));
		ob_start();
		try {
			include $filename;
		}
		catch (Exception $e) { // we should not let an exception kill Joomla!
			ob_end_clean();
			if ($this->show_errors) {
				return '<strong>JIncludes (_renderExtPHP) caught exception: ' 
					.$e->getMessage(). '<strong>';
			} else {
				return $env[0];
			}
		}
		$output = ob_get_contents();
		ob_end_clean();
		return $output;
	}
	
	/**
	 * Attaches a stylesheet link to the <head> of the current document
	 *
	 * @param int $id the snippet being rendered
	 *
	 * @return void
	 */
	protected function _attachStylesheet( $id )
	{
		$doc =& JFactory::getDocument();
		$doc->addStyleSheet(trim($this->params->get("code$id", '')));
	}
	
	/**
	 * Attaches an external script file to the <head> of the current document
	 *
	 * @param int $id the snippet being rendered
	 *
	 * @return void
	 */
	protected function _attachScript( $id )
	{
		$doc =& JFactory::getDocument();
		$doc->addScript(trim($this->params->get("code$id", '')));
	}
	
}